<?php
/**
 * FPDF Graph - FPDF class extension
 * Copyright (c) 2005-2010, http://www.interpid.eu
 *
 * FPDF Table is licensed under the terms of the GNU Open Source GPL 3.0
 * license.
 *
 * Commercial use is prohibited. Visit <http://www.interpid.eu/fpdf-components>
 * if you need to obtain a commercial license.
 *
 * This program is free software: you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation, either version 3 of the License, or any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program. If not, see <http://www.gnu.org/licenses/gpl.html>.
 *
 * 
 * Version:       1.0
 * Date:          2007/07/01
 * Author:        Bintintan Andrei <andy@interpid.eu>
 */

require_once("class.fpdf.php");

require_once('class.bezier.php');
    
    
class fpdf_graph extends FPDF{
    
/*****************************************************************************************************************************************************************
 * 
 * GRAPH CLASS
 * 
 * 
 ****************************************************************************************************************************************************************/

/**
 * Graph text multicell width
 */
var    $GRAPH_TEXT_MWIDTH = 10;


/**
 * Graph text multicell height
 */
var    $GRAPH_TEXT_MHEIGHT = 2;

/**
* @desc     Start X position of the graph. Upper Left Corner X
* @access   var
*/
var    $iGraphStartX = 0;

/**
* @desc     Start Y position of the graph. Upper Left Corner Y
* @access   var
*/
var    $iGraphStartY = 0;

/**
* @desc     Graph Width
* @access   var
*/
var    $iGraphWidth = 100;

/**
* @desc     Graph Height
* @access   var
*/
var    $iGraphHeight = 100;


/**
* @desc     The value of the X for the Left side of the graph. (The minimum X that will be represented)
* @access   var
*/
var    $fGraphXMin = 0;
var    $fGraphXMax = 10;
var    $fGraphYMin = 0;
var    $fGraphYMax = 10;

var    $iGraphGridLinesX = 0;//The number or Value of X grid lines 
var    $iGraphGridLinesY = 0;//The number or Value of Y grid lines 

var    $bGraphGridShow = true;
var    $bGraphGridValuesShow = true;

var    $bGraphBorderShow = true;

var    $aGraphParams = array();

var    $sGraphTextX = '';
var    $sGraphTextY = '';

var    $aGraphTexts = array();


/**
 * Loas the Graphic settings
 *
 * @access  var
 * @param   array - $aParams - associative array that contains the graph settings.
 * @return  void
 */
function GraphLoadSettings($aParams){

    if (isset($aParams['BORDER_SHOW']))     $this->bGraphBorderShow = $aParams['BORDER_SHOW'];
    if (isset($aParams['GRID_SHOW']))         $this->bGraphGridShow = $aParams['GRID_SHOW'];
    if (isset($aParams['GRID_LINESX']))     $this->iGraphGridLinesX = $aParams['GRID_LINESX'];
    if (isset($aParams['GRID_LINESY']))     $this->iGraphGridLinesY = $aParams['GRID_LINESY'];
    if (isset($aParams['WIDTH']))             $this->iGraphWidth = $aParams['WIDTH'];
    if (isset($aParams['HEIGHT']))             $this->iGraphHeight = $aParams['HEIGHT'];
    
    $this->aGraphParams = $aParams;
    
}//GraphLoadSettings


/**
 * Initializes the Graph Variables
 *
 * @param   none
 * @return  void
 */
function GraphInitialize(){
    $this->aGraphTexts = array();
}

/**
 * Sets the Graph Coordonates(X and Y)/Width and Height
 *
 * @param    numeric $iGraphStartX - Start X Position(relative to the page)
 * @param    numeric $iGraphStartY - Start Y Position(relative to the page)
 * @param    numeric $iGraphWidth - Graph Widht
 * @param    numeric $iGraphHeight - Graph Height
 */
function SetGraphCoordonates($iGraphStartX, $iGraphStartY, $iGraphWidth, $iGraphHeight){
    $this->iGraphStartX = $iGraphStartX;
    $this->iGraphStartY = $iGraphStartY;
    $this->iGraphWidth = $iGraphWidth;
    $this->iGraphHeight = $iGraphHeight;
}//SetGraphCoordonates

/**
 * Sets the Graph Limits, Minimum and Maximum values
 *
 * @access   public
 * @param    numeric $fGraphXMin - X Minim
 * @param    numeric $fGraphXMax - X Maxim
 * @param    numeric $fGraphYMin - Y Minim
 * @param    numeric $fGraphYMax - Y Maxim
 */
public function SetGraphLimits($fGraphXMin, $fGraphXMax, $fGraphYMin, $fGraphYMax){
    $this->fGraphXMin = $fGraphXMin;
    $this->fGraphXMax = $fGraphXMax;
    $this->fGraphYMin = $fGraphYMin;
    $this->fGraphYMax = $fGraphYMax;
}//SetGraphLimits

/**
 * Sets the Grid Parameters if there are any specified
 *
 * @param    numeric $iLineWidth - the Line Width
 * @param    array $aColor - color of the line eg: array(10, 200, 124) RGB combination
 * @param    boolean $bShow - ==true the Grid is showed
 * @param    numeric $iCountX - Number of Grid Lines X
 * @param    numeric $iCountY - Number of Grid Lines Y
 */
function SetGridParams($iLineWidth = null, $aColor = null, $bShow = null, $iCountX = null, $iCountY = null){
    if ($iLineWidth != null) $this->aGraphParams['GRID_SIZE'] = $iLineWidth;
    if (is_array($aColor)) $this->aGraphParams['GRID_COLOR'] = $aColor;
    if ($bShow != null) $this->bGraphGridShow = (bool) $bShow;
    if ($iCountX != null) $this->iGraphGridLinesX = (int) $iCountX;
    if ($iCountY != null) $this->iGraphGridLinesY = (int) $iCountY;
}//SetGridParams

function GraphDrawGridLine($fCurVal, $sLineType){
    
    switch (strtoupper($sLineType)){
        case 'X':
                $y1 = $this->iGraphStartY;
                $y2 = $this->iGraphStartY + $this->iGraphHeight;

                $x1 = $this->GraphValuesToCoords($fCurVal, 'x');
                $x2 = $x1;
                
                $this->SetXY($x1 - $this->GRAPH_TEXT_MWIDTH / 2, $y2 + 1);
                
                $txtVal = round($fCurVal, 2);
                
                $this->MultiCell($this->GRAPH_TEXT_MWIDTH , $this->GRAPH_TEXT_MHEIGHT , $txtVal, '', 'C');        
                
                $this->Line($x1, $y1, $x2, $y2);
            break;
        case 'Y':
                $x1 = $this->iGraphStartX;
                $x2 = $this->iGraphStartX + $this->iGraphWidth;            
                
                $y1 = $this->GraphValuesToCoords($fCurVal, 'y');
                $y2 = $y1;
                
                $yPos = $x1 - $this->GRAPH_TEXT_MWIDTH - 1;
                //this value must be bigger than 0 !!!
                if ( $yPos < 0 ) $yPos = 0;
                
                $this->SetXY($yPos, $y1 - $this->GRAPH_TEXT_MHEIGHT / 2);
                
                $txtVal = round($fCurVal, 2);            
                
                $this->MultiCell($this->GRAPH_TEXT_MWIDTH, $this->GRAPH_TEXT_MHEIGHT , $txtVal, '', 'R');
                
                $this->Line($x1, $y1, $x2, $y2);
            break;
    }
}//GraphDrawGridLine

function GraphDrawGrid($aParam = array()){
    
    $this->iGraphStartX = $this->getX();
    $this->iGraphStartY = $this->getY();
        

    //Draw the background
    if ($this->bGraphBorderShow){ 
                
        $iLineWidth    = $this->GraphExtractParameters('BORDER_SIZE', $aParam);
        $aBorderColor = $this->GraphExtractParameters('BORDER_COLOR', $aParam);
        $aBackColor = $this->GraphExtractParameters('BG_COLOR', $aParam);
        
        $this->SetLineWidth($iLineWidth);
        $this->SetDrawColor($aBorderColor[0], $aBorderColor[1], $aBorderColor[2]);
        $this->SetFillColor($aBackColor[0], $aBackColor[1], $aBackColor[2]);
        
        $this->Rect($this->iGraphStartX, $this->iGraphStartY, $this->iGraphWidth, $this->iGraphHeight, "F");        
    }

    //Draw the grid lines
    
    $tSize         = $this->GraphExtractParameters('GRID_TEXT_SIZE', $aParam);
    $tFont         = $this->GraphExtractParameters('GRID_TEXT_FONT', $aParam);
    $tFontType     = $this->GraphExtractParameters('GRID_TEXT_FONTTYPE', $aParam);
    $tColor     = $this->GraphExtractParameters('GRID_TEXT_COLOR', $aParam);

    $this->SetTextColor($tColor[0], $tColor[1], $tColor[2]);
    $this->SetFont($tFont, $tFontType, $tSize);    
    
    
    $bGridLineByValue = $this->GraphExtractParameters('GRID_LINES_BY_VALUE', $aParam);    
    
    $aGridColor        = $this->GraphExtractParameters('GRID_COLOR', $aParam);
    $iGridLineWidth = $this->GraphExtractParameters('GRID_SIZE', $aParam);

    $this->SetLineWidth($iGridLineWidth);
    $this->SetDrawColor($aGridColor[0], $aGridColor[1], $aGridColor[2]);
    
    
    if (true == $bGridLineByValue){
        /**
         * GRID LINES BY VALUE
         */
        //vertical grids X Lines
        if ($this->iGraphGridLinesX > 0){
            
            $fCurVal = 0;
            while ($fCurVal > $this->fGraphXMin){        
                
                if (($fCurVal > $this->fGraphXMin) && (($fCurVal < $this->fGraphXMax))) $this->GraphDrawGridLine($fCurVal, 'x');            
                $fCurVal -= $this->iGraphGridLinesX;
            }//while
            
            $fCurVal = $this->iGraphGridLinesX;
            while ($fCurVal < $this->fGraphXMax){        
                
                if (($fCurVal > $this->fGraphXMin) && (($fCurVal < $this->fGraphXMax))) $this->GraphDrawGridLine($fCurVal, 'x');    
                $fCurVal += $this->iGraphGridLinesX;
            }//while
        }//fi

        //vertical grids Y Lines
        if ($this->iGraphGridLinesY > 0){
                                
            $fCurVal = 0;
            while ($fCurVal > $this->fGraphYMin){        
                
                if (($fCurVal > $this->fGraphYMin) && (($fCurVal < $this->fGraphYMax))) $this->GraphDrawGridLine($fCurVal, 'y');            
                $fCurVal -= $this->iGraphGridLinesY;
            }//while
            
            $fCurVal = $this->iGraphGridLinesY;
            while ($fCurVal < $this->fGraphYMax){        
                
                if (($fCurVal > $this->fGraphYMin) && (($fCurVal < $this->fGraphYMax))) $this->GraphDrawGridLine($fCurVal, 'y');            
                $fCurVal += $this->iGraphGridLinesY;
            }//while
        }//fi
        
    }else{
        /**
         * GRID LINES BY COUNT
         */
        
        if ($this->iGraphGridLinesX > 0){
            
            $iGrids = $this->iGraphGridLinesX + 1;
            $fDist = ($this->iGraphWidth / $iGrids);
            $fVal = ($this->fGraphXMax - $this->fGraphXMin) / $iGrids;
            
            $y1 = $this->iGraphStartY;
            $y2 = $this->iGraphStartY + $this->iGraphHeight;        
                    
            for ($i = 0; $i < $this->iGraphGridLinesX; $i++){
                
                $x1 = $this->iGraphStartX + $fDist * ($i + 1);
                $x2 = $x1;
                
                $this->SetXY($x1 - $this->GRAPH_TEXT_MWIDTH / 2, $y2 + 1);
                
                $txtVal = $this->fGraphXMin + $fVal * ($i + 1);
                $txtVal = round($txtVal, 2);
                
                $this->MultiCell($this->GRAPH_TEXT_MWIDTH , $this->GRAPH_TEXT_MHEIGHT , $txtVal, '', 'C');
                
                $this->Line($x1, $y1, $x2, $y2);
                
            }//for
        }//fi
        
        //horizontal grids Y Lines
        if ($this->iGraphGridLinesY > 0){
            
            $iGrids = $this->iGraphGridLinesY + 1;
            $fDist = ($this->iGraphHeight / $iGrids);
            $fVal = ($this->fGraphYMax - $this->fGraphYMin) / $iGrids;
            
            $x1 = $this->iGraphStartX;
            $x2 = $this->iGraphStartX + $this->iGraphWidth;
                    
            for ($i = 0; $i < $this->iGraphGridLinesY; $i++){
                            
                $y1 = $this->iGraphStartY + $fDist * ($i + 1);
                $y2 = $y1;
                
                $yPos = $x1 - $this->GRAPH_TEXT_MWIDTH - 1;
                //this value must be bigger than 0 !!!
                if ( $yPos < 0 ) $yPos = 0;
                
                $this->SetXY($yPos, $y1 - $this->GRAPH_TEXT_MHEIGHT / 2);
                
                $txtVal = $this->fGraphYMax - $fVal * ($i + 1);
                $txtVal = round($txtVal, 2);            
                
                $this->MultiCell($this->GRAPH_TEXT_MWIDTH, $this->GRAPH_TEXT_MHEIGHT , $txtVal, '', 'R');
                
                $this->Line($x1, $y1, $x2, $y2);
                
            }//for
        }//fi
    }
    
    $tSize         = $this->GraphExtractParameters('GRAPH_TEXT_SIZE', $aParam);
    $tFont         = $this->GraphExtractParameters('GRAPH_TEXT_FONT', $aParam);
    $tFontType     = $this->GraphExtractParameters('GRAPH_TEXT_FONTTYPE', $aParam);
    $tColor     = $this->GraphExtractParameters('GRAPH_TEXT_COLOR', $aParam);
    
    $this->SetTextColor($tColor[0], $tColor[1], $tColor[2]);
    $this->SetFont($tFont, $tFontType, $tSize);    

    
    if ( strlen($this->sGraphTextX) > 0 ){
        $this->SetXY($this->iGraphStartX, $this->iGraphStartY + $this->iGraphHeight + $this->GRAPH_TEXT_MHEIGHT * 1.8);
        $this->MultiCell($this->iGraphWidth, $this->GRAPH_TEXT_MHEIGHT , $this->sGraphTextX, '', 'C');
    }
    
    if ( strlen($this->sGraphTextY) > 0 ){
        $this->SetXY($this->iGraphStartX, $this->iGraphStartY - $this->GRAPH_TEXT_MHEIGHT - 1);
        $this->MultiCell($this->iGraphWidth, $this->GRAPH_TEXT_MHEIGHT , $this->sGraphTextY, '', 'L');
    }
    
    
    
    if ($this->bGraphBorderShow){ 
                
        $iLineWidth    = $this->GraphExtractParameters('BORDER_SIZE', $aParam);
        $aBorderColor = $this->GraphExtractParameters('BORDER_COLOR', $aParam);
        $aBackColor = $this->GraphExtractParameters('BG_COLOR', $aParam);
        
        $this->SetLineWidth($iLineWidth);
        $this->SetDrawColor($aBorderColor[0], $aBorderColor[1], $aBorderColor[2]);
        $this->Rect($this->iGraphStartX, $this->iGraphStartY, $this->iGraphWidth, $this->iGraphHeight);        
    }

    
}


/**
 * Returns the Value of the given Coordonate
 *
 * @param    float $iVal - the value that is given to be determined
 * @param    string $sCoordType - X or Y
 * @return   the Value of the Coordonate
 */
function GraphCoordsToValues($iVal, $sCoordType){
    $min = 0;
    $max = 0;
    $units = 0;
    switch(strtoupper($sCoordType)){
        case 'X':
            $min = $this->fGraphXMin;
            $max = $this->fGraphXMax;
            $units = $this->iGraphWidth;
            break;
        
        case 'Y':
            $min = $this->fGraphYMin;
            $max = $this->fGraphYMax;
            $units = $this->iGraphHeight;
            break;
    }
    
    $retVal = (($max - $min) / $units ) * $iVal;
    
    return $retVal;
}//function GraphCoordsToValues

/**
 * Returns the Coordonate on the Graph for the given Value
 *
 * @param    float $iVal - the value that is given to be determined
 * @param    string $sCoordType - X or Y
 * @return   the Coordonate of the Value
 */
function GraphValuesToCoords($iVal, $sCoordType){
    $min = 0;
    $max = 0;
    $units = 0;
    $start = 0;
    $sign = 1;
    
    switch(strtoupper($sCoordType)){
        case 'X':
            $min = $this->fGraphXMin;
            $max = $this->fGraphXMax;
            $units = $this->iGraphWidth;
            $start = $this->iGraphStartX;
            $sign = 1;
            break;
        
        case 'Y':
            $min = $this->fGraphYMin;
            $max = $this->fGraphYMax;
            $units = $this->iGraphHeight;
            $start = $this->iGraphStartY + $this->iGraphHeight;
            $sign = -1;
            break;
    }
        
    $retVal = $start + $sign * ($units / ($max - $min)) * ($iVal - $min);
    
    return $retVal;    
}


/**
 * Draws a Graphic or Diagram
 *
 * @param    array $aCoords - contains the coorodnates
 * @param    string $sText - Graph Text
 * @param    array $aParams - array that contains parameters
 * @return   void
 */
function GraphDraw($aCoords, $sText = '', $aParams = array()){
    
    $aDiagColor = $this->GraphExtractParameters('DIAGRAM_COLOR', $aParams);
    $iLineWidth = $this->GraphExtractParameters('DIAGRAM_SIZE', $aParams);
    
    $this->SetDrawColor($aDiagColor[0], $aDiagColor[1], $aDiagColor[2]);
    $this->SetLineWidth($iLineWidth);
    
    //eliminate the NULL values
    foreach ($aCoords as $key => $val){
        if (($key === null) || ($val === null))
            unset($aCoords[$key]);
    }
    
    $aX = array_keys($aCoords);
    $aY = array_values($aCoords);
    
    $bezier = new Bezier($aX, $aY);
    
    $step = 0.01;
    $start = 0;
    $end = 1.01;
    $current = $start;
    
    $aPosLast = $bezier->GetPoint($start);
    
    do {
        $aPosCurrent = $bezier->GetPoint($current);
                
        $this->GraphDrawLine($aPosLast[0], $aPosLast[1], $aPosCurrent[0], $aPosCurrent[1]);
        
        $aPosLast = $aPosCurrent;
        
        $current += $step;
        
    }while ($current < $end);
    
    //put the graph text if text is received
    if ($sText != ''){
        $this->aGraphTexts[] = array(
            'x' => $this->GraphValuesToCoords($aPosCurrent[0], 'x'),
            'y' => $this->GraphValuesToCoords($aPosCurrent[1], 'y'),
            'text' => $sText
        );
        
    }
    //$this->GraphPutText($aPosCurrent[0], $aPosCurrent[1], $sText);
        
}//GraphDraw

/**
 * Outputs the Graph Text for all the lines
 *
 * @access    private
 * @param     none
 * @return    void
 */
function GraphPutAllText(){
    
    /**
     * $aGT = &$this->aGraphTexts;
     * 
     */
    $aGT = $this->aGraphTexts;//keep it without reference for PHP 4!!! BUG in array multisort.
    $nr = count($aGT);
    
    //sort the array by the Y
    
    $aY = array();
    
    for ($i=0; $i< $nr; $i++){
        $aY[] = $aGT[$i]['y'];
    }
    
    array_multisort($aY, SORT_ASC, $aGT);
        
    for ($i=0; $i < $nr-1; $i++){
        
        $diffX = $aGT[$i+1]['x'] - $aGT[$i]['x'];
        $diffY = $aGT[$i+1]['y'] - $aGT[$i]['y'];
        
        if (($diffX < $this->GRAPH_TEXT_MWIDTH) && ($diffY < $this->GRAPH_TEXT_MHEIGHT)){
            $aGT[$i+1]['y'] += $this->GRAPH_TEXT_MHEIGHT - $diffY;
        }
    }
    
    foreach ($aGT as $val){
        $this->GraphPutText($val['x'], $val['y'], $val['text']);
    }
}//GraphPutAllText

/**
 * Outputs the Text for the Current Graph Line
 *
 * @access   private
 * @param    float $x - Graph X Value
 * @param    float $y - Graph Y Value
 * @param    string $sText - Text
 * @param    array $aParam - parameters
 * @return   void
 */
function GraphPutText($x, $y, $sText, $aParam = array()){
    
    
    $tSize         = $this->GraphExtractParameters('DIAGRAM_TEXT_SIZE', $aParam);
    $tFont         = $this->GraphExtractParameters('DIAGRAM_TEXT_FONT', $aParam);
    $tFontType     = $this->GraphExtractParameters('DIAGRAM_TEXT_FONTTYPE', $aParam);
    $tColor     = $this->GraphExtractParameters('DIAGRAM_TEXT_COLOR', $aParam);

    $this->SetTextColor($tColor[0], $tColor[1], $tColor[2]);
    $this->SetFont($tFont, $tFontType, $tSize);
    
    $this->SetXY($x, $y - ($this->GRAPH_TEXT_MHEIGHT / 2));
    
    $this->MultiCell(0, $this->GRAPH_TEXT_MHEIGHT, $sText);
    
}//GraphPutText

/**
 * Draws a Graph Line. The received data are values that are converted to coordonates.
 *
 * @access    protected
 * @param     float $x0 - X0 value
 * @param     float $y0 - Y0 value
 * @param     float $x1 - X1 value
 * @param     float $y1 - Y1 valuye
 * @return    none
 */
function GraphDrawLine($x0, $y0, $x1, $y1){
        
    $x0 = $this->GraphValuesToCoords($x0, 'x');
    $x1 = $this->GraphValuesToCoords($x1, 'x');
    $y0 = $this->GraphValuesToCoords($y0, 'y');
    $y1 = $this->GraphValuesToCoords($y1, 'y');
    
    $this->Line($x0, $y0, $x1, $y1);
}//GraphDrawLine

/**
 * Extracts the specified parameter value.
 * First looks in the specified array if given, otherwise it looks in the default parameter array of the class
 * If the specified parameter is not found NULL is returned or the specified $default value
 *
 * @access   protected
 * @param    string $sParameter - parameter name
 * @param    array $aParams - optional parameter array
 * @param    value $default - default value if the parameter was not found
 * @return   the parameter value if found or default
 */
function GraphExtractParameters($sParameter, $aParams = array(), $default = null){
    $aGraphParams = $this->aGraphParams;
    
    $retVal = $default;
    
    if (isset($aParams[$sParameter])) $retVal = $aParams[$sParameter];
    elseif (isset($aGraphParams[$sParameter])) $retVal = $aGraphParams[$sParameter];

    return $retVal;
}//GraphExtractParameters

/**
 * Graphic/Diagram draw function
 *
 * @access   public
 * @param    array $ArrWerte - contains the X and Y points
 * @param    string $szBeschriftungX - Description X
 * @param    string $szBeschriftungY - Descrition Y
 * @param    integer $x - Start X Position
 * @param    integer $y - Start Y Position
 * @param    integer $w - Graph Width
 * @param    integer $h - Graph Height
 * @param    array $aParams - Graph Parameters
 * @return   void
 */
function GraphDiagramm($ArrWerte, $szBeschriftungX='', $szBeschriftungY='', $x=null, $y=null, $w=null, $h=null, $aParams=array()){
    
    $this->GraphInitialize();
    
    if (( $x != null ) && ( $y != null )) $this->SetXY($x, $y);
    
    if ( $w != null ) $this->iGraphWidth = $w;
    if ( $h != null ) $this->iGraphHeight = $h;
    
    
    $this->sGraphTextX = $szBeschriftungX;
    $this->sGraphTextY = $szBeschriftungY;
    
    $this->GraphDrawGrid($aParams);
    
    $aXCoords = $ArrWerte['x'];
    
    unset($ArrWerte['x']);
        
    foreach ($ArrWerte as $key => $val){
        if (isset($aParams[$key]))    $params = $aParams[$key];
        else $params = array();
        
        $res = array();
        
        foreach($aXCoords as $key1 => $val1){
            $res[$val1] = $val[$key1];
        }
        
        $this->GraphDraw($res, $key, $params);
    }
    
    $this->GraphPutAllText();
}//Graph_td_diagramm

}//class

?>